<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CourseController extends Controller
{
    public function index()
    {
        $courses = Course::withCount('enrollments')->latest()->get();
        return view('admin.courses.index', compact('courses'));
    }

    public function create()
    {
        return view('admin.courses.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title'          => ['required', 'string', 'max:255'],
            'code'           => ['required', 'string', 'max:20', 'unique:courses,code'],
            'type'           => ['required', 'in:diploma,artisan'],
            'duration_weeks' => ['required', 'integer', 'min:1'],
            'fee'            => ['required', 'numeric', 'min:0'],
            'description'    => ['nullable', 'string'],
            'is_active'      => ['nullable', 'boolean'],
        ]);

        $data['slug']      = Str::slug($data['title']);
        $data['is_active'] = $request->boolean('is_active', true);

        Course::create($data);

        return redirect()->route('admin.courses.index')
            ->with('success', 'Course created successfully.');
    }

    public function edit(Course $course)
    {
        return view('admin.courses.edit', compact('course'));
    }

    public function update(Request $request, Course $course)
    {
        $data = $request->validate([
            'title'          => ['required', 'string', 'max:255'],
            'code'           => ['required', 'string', 'max:20', 'unique:courses,code,' . $course->id],
            'type'           => ['required', 'in:diploma,artisan'],
            'duration_weeks' => ['required', 'integer', 'min:1'],
            'fee'            => ['required', 'numeric', 'min:0'],
            'description'    => ['nullable', 'string'],
            'is_active'      => ['nullable', 'boolean'],
        ]);

        $data['slug']      = Str::slug($data['title']);
        $data['is_active'] = $request->boolean('is_active');

        $course->update($data);

        return redirect()->route('admin.courses.index')
            ->with('success', 'Course updated successfully.');
    }

    public function destroy(Course $course)
    {
        $course->delete();

        return redirect()->route('admin.courses.index')
            ->with('success', 'Course deleted.');
    }
}
