<?php

namespace App\Services;

use App\Models\Certificate;
use App\Models\Enrollment;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CertificateService
{
    /**
     * Generate a certificate number in the format:
     * CERT/{YEAR}/{COURSE_CODE}/{SEQUENCE}
     * e.g. CERT/2026/DIPIT/00045
     */
    public function generateCertificateNumber(Enrollment $enrollment): string
    {
        $year       = now()->year;
        $courseCode = strtoupper($enrollment->course->code ?? 'GEN');
        $count      = Certificate::whereYear('issued_at', $year)->count() + 1;
        $sequence   = str_pad($count, 5, '0', STR_PAD_LEFT);

        return "CERT/{$year}/{$courseCode}/{$sequence}";
    }

    /**
     * Issue a certificate for a completed enrollment.
     * Marks the enrollment as completed and creates the certificate record.
     * Throws an exception if a certificate already exists.
     */
    public function issue(Enrollment $enrollment, int $adminId): Certificate
    {
        if ($enrollment->certificate) {
            throw new \RuntimeException('A certificate has already been issued for this enrollment.');
        }

        return DB::transaction(function () use ($enrollment, $adminId) {

            // Mark enrollment as completed
            $enrollment->update([
                'status'       => 'completed',
                'completed_at' => now(),
            ]);

            $certNumber = $this->generateCertificateNumber($enrollment);

            return Certificate::create([
                'enrollment_id'      => $enrollment->id,
                'certificate_number' => $certNumber,
                'verification_code'  => (string) Str::uuid(),
                'issued_at'          => now(),
                'created_by'         => $adminId,
            ]);
        });
    }
}
